CREATE OR REPLACE PACKAGE SVCO.vcontrol authid current_user
IS
  /*
  * "Schema Version Control for Oracle" (SVCO)
  *      http://www.sumsoftsolutions.com
  * PLEASE READ LICENSE AGREEMENT CAREFULLY BEFORE USING THIS SOFTWARE
  *
  * PURPOSE
  * This package is the user interface to Schema Version Control for Oracle functionality.
  *
  * SECURITY
  * This package is granted (EXECUTE rights) to SVCO_REP_OPERATOR and SVCO_REP_ADMIN roles.
  */
  --
  -- DESCRIPTION
  -- This function return current SVCO version.
  --
  -- PARAMETERS
  -- none
  --
  -- EXCEPTIONS
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  FUNCTION Get_SVCO_Version
    RETURN VARCHAR2;
  --
  -- DESCRIPTION
  -- This procedure saves purchased LICENSE KEY for SVCO product.
  --
  -- PARAMETERS
  -- Key - a purchased LICENSE KEY
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- This procedure performs a transaction.
  --
  PROCEDURE Save_License_Key(
      KEY IN CHAR);
  --
  -- DESCRIPTION
  -- This function return 1 if SVCO is started or 0 - if not.
  --
  -- PARAMETERS
  -- none
  --
  -- EXCEPTIONS
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  FUNCTION SVCO_Started_To_Number
    RETURN NUMBER;
  --
  -- DESCRIPTION
  -- This function return TRUE if SVCO is started or FALSE - if not.
  --
  -- PARAMETERS
  -- none
  --
  -- EXCEPTIONS
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  FUNCTION SVCO_Started
    RETURN BOOLEAN;
  --
  -- DESCRIPTION
  -- This function return 1 if SVCO is enabled or 0 - if not.
  --
  -- PARAMETERS
  -- none
  --
  -- EXCEPTIONS
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  FUNCTION SVCO_Enabled_To_Number
    RETURN NUMBER;
  --
  -- DESCRIPTION
  -- This function return TRUE if SVCO is enabled or FALSE - if not.
  --
  -- PARAMETERS
  -- none
  --
  -- EXCEPTIONS
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  FUNCTION SVCO_Enabled
    RETURN BOOLEAN;
  --
  -- DESCRIPTION
  -- This procedure tries to enable SVCO.
  --
  -- PARAMETERS
  -- none
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  PROCEDURE Enable_Version_Control;
  --
  -- DESCRIPTION
  -- This procedure tries to disable SVCO.
  --
  -- PARAMETERS
  -- none
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  PROCEDURE Disable_Version_Control;
  --
  -- DESCRIPTION
  -- This function return 1 if role SVCO_REP_ADMIN is granted to session user or 0 - if not.
  --
  -- PARAMETERS
  -- none
  --
  -- EXCEPTIONS
  -- none
  --
  -- TRANSACTION
  -- No.
  --none
  FUNCTION Has_Admin_Role
    RETURN NUMBER;
  --
  -- DESCRIPTION
  -- This procedure tries to create new objects versions for given Schema_Name.
  --
  -- PARAMETERS
  -- Schema_Name - Schema name (user name) for which you create a new revision,
  --               if not given - perform operation for all known schemas in repository.
  --               Case insensitive.
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- This procedure performs a transaction.
  --
  PROCEDURE Repository_Sync(
      Schema_Name IN VARCHAR2);
  --
  -- DESCRIPTION
  -- This procedure returns a DDL script for the given object version.
  --
  -- PARAMETERS
  -- Schema_Name - Schema name (user name). Case insensitive.
  --
  -- Object_Name - Object name. Case sensitive!
  --
  -- Object_Type - Object type. Case insensitive.
  --
  -- DDL - DDL script.
  --
  -- Version_ID - Object version number, if not given - last version.
  --
  -- New_Schema_Name - To be given when all objects are prefixed
  --                   with this schema name instead of original. Case insensitive.
  --
  -- With_Header - 'YES' indicate that DDL script includes SVCO header.
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_too_many_rows
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_req_param
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  PROCEDURE Get_Object_DDL(
      Schema_Name IN VARCHAR2 ,
      Object_Name IN VARCHAR2 ,
      Object_Type IN VARCHAR2 ,
      DDL OUT CLOB ,
      Version_ID      IN NUMBER DEFAULT NULL ,
      New_Schema_Name IN VARCHAR2 DEFAULT NULL ,
      With_Header     IN VARCHAR2 DEFAULT 'YES');
  --
  -- DESCRIPTION
  -- This procedure returns a DDL script for the given dropped object version.
  --
  -- PARAMETERS
  -- Dropped_Object_ID - Dropped Object ID.
  --
  -- DDL - DDL script.
  --
  -- Version_ID - Object version number, if not given - last version.
  --
  -- New_Schema_Name - To be given when all objects are prefixed
  --                   with this schema name instead of original. Case insensitive.
  --
  -- With_Header - 'YES' indicate that DDL script includes SVCO header.
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_too_many_rows
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_req_param
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  PROCEDURE Get_Dropped_Object_DDL(
      Dropped_Object_ID IN NUMBER ,
      DDL OUT CLOB ,
      Version_ID      IN NUMBER DEFAULT NULL ,
      New_Schema_Name IN VARCHAR2 DEFAULT NULL ,
      With_Header     IN VARCHAR2 DEFAULT 'YES');
  --
  -- DESCRIPTION
  -- This procedure delete the object version from repository.
  --
  -- PARAMETERS
  -- Schema_Name - Schema name (user name). Case insensitive.
  --
  -- Object_Name - Object name. Case insensitive.
  --
  -- Object_Type - Object type. Case insensitive.
  --
  -- Version_ID - Object version number.
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- This procedure performs a transaction.
  --
  PROCEDURE Delete_Object_Version(
      Schema_Name IN VARCHAR2 ,
      Object_Name IN VARCHAR2 ,
      Object_Type IN VARCHAR2 ,
      Version_ID  IN NUMBER);
  --
  -- DESCRIPTION
  -- This procedure delete the dropped object version from repository.
  --
  -- PARAMETERS
  -- Dropped_Object_ID - Dropped Object ID.
  --
  -- Version_ID - Object version number.
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- This procedure performs a transaction.
  --
  PROCEDURE Delete_Dropped_Object_Version(
      Dropped_Object_ID IN NUMBER ,
      Version_ID        IN NUMBER);
  --
  -- DESCRIPTION
  -- This procedure create a new tag in repository.
  --
  -- PARAMETERS
  -- Tag_Name - The name of a tag. Must be unique.
  --
  -- Comments - Tag comments.
  --
  -- EXCEPTIONS
  -- exceptions_.exc_too_many_rows
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- This procedure performs a transaction.
  --
  PROCEDURE Create_New_Tag(
      Tag_Name IN VARCHAR2 ,
      Comments IN VARCHAR2);
  --
  -- DESCRIPTION
  -- This procedure delete an existing tag from repository.
  --
  -- PARAMETERS
  -- Tag_Name - The name of a tag. Case sensitive.
  --
  -- Untag - Default 'NO', 'YES' if should perform untag before deleting.
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- This procedure performs a transaction.
  --
  PROCEDURE Delete_Tag(
      Tag_Name IN VARCHAR2,
      Untag    IN VARCHAR2 DEFAULT 'NO');
  --
  -- DESCRIPTION
  -- This procedure tag corresponding object version with given tag.
  --
  -- PARAMETERS
  -- Tag_Name - The name of a tag. Case sensitive.
  --
  -- Schema_Name - Schema name (user name). Case insensitive.
  --
  -- Object_Name - Object name. Case insensitive. Wildcards allowed.
  --
  -- Object_Type - Object type. Case insensitive. Wildcards allowed.
  --
  -- Version_ID - Object version number, if not givvent tag the latest object version
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- This procedure performs a transaction.
  --
  PROCEDURE Tag_Object_Version(
      Tag_Name    IN VARCHAR2 ,
      Schema_Name IN VARCHAR2 ,
      Object_Name IN VARCHAR2 ,
      Object_Type IN VARCHAR2 ,
      Version_ID  IN NUMBER DEFAULT NULL);
  --
  -- DESCRIPTION
  -- This procedure untag corresponding object version with given tag.
  --
  -- PARAMETERS
  -- Tag_Name - The name of a tag. Case sensitive.
  --
  -- Schema_Name - Schema name (user name). Case insensitive.
  --
  -- Object_Name - Object name. Case insensitive. Wildcards allowed.
  --
  -- Object_Type - Object type. Case insensitive. Wildcards allowed.
  --
  -- Version_ID - Object version number, if not givvent untag all object versions
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- This procedure performs a transaction.
  --
  PROCEDURE Untag_Object_Version(
      Tag_Name    IN VARCHAR2 ,
      Schema_Name IN VARCHAR2 ,
      Object_Name IN VARCHAR2 ,
      Object_Type IN VARCHAR2 ,
      Version_ID  IN NUMBER DEFAULT NULL);
  --
  -- DESCRIPTION
  -- This procedure returns a DDL script for objects with given tag.
  --
  -- PARAMETERS
  -- Tag_Name - The name of a tag. Case sensitive.
  --
  -- DDL - DDL script.
  --
  -- Schema_Name - Schema name (user name). Case insensitive. Wildcards allowed.
  --
  -- Object_Name - Object name. Case sensitive! Wildcards allowed.
  --
  -- Object_Type - Object type. Case insensitive. Wildcards allowed.
  --
  -- New_Schema_Name - To be given when all objects are prefixed
  --                   with this schema name instead of original. Case insensitive.
  --
  -- With_Header - 'YES' indicate that DDL script includes SVCO header.
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_req_param
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  PROCEDURE Get_Objects_DDL_By_Tag(
      Tag_Name IN VARCHAR2 ,
      DDL OUT CLOB ,
      Schema_Name     IN VARCHAR2 DEFAULT '%',
      Object_Name     IN VARCHAR2 DEFAULT '%',
      Object_Type     IN VARCHAR2 DEFAULT '%',
      New_Schema_Name IN VARCHAR2 DEFAULT NULL ,
      With_Header     IN VARCHAR2 DEFAULT 'YES');
END vcontrol;
/
